#include "lrqElib.h"
#include "lrqEdefs.h"
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef lrqEread
# undef lrqEread
#endif
#undef read

int
lrqEread(descriptor, buffer, nbytes)
    int descriptor;
    char *buffer;
    unsigned int nbytes;
{
    int retval;
    int savsyserrno;
    char *reason;
    int ewhat;
    int flags;

    errno = 0;

    if ((retval = read(descriptor, buffer, nbytes)) >= 0) {
	return retval;
    }
    /* we have an error */

    /* by default, don't exit from read() */
    ewhat = lrqE_severity & (~lrqE_EFATAL);

    /* Save the original error message */
    savsyserrno = errno;

    switch (errno) {
    case EAGAIN: /* not a fatal error at all! */
	if (fcntl(descriptor, F_GETFL, 0) & O_NDELAY) {
	    return -1;
	    /* If this happens, it is because you are using file locks
	     * (see chmod(2)) with NDELAY set, and you ought to be expecting it!
	     */
	}
	reason = "not enough internal system memory [EAGAIN]";
	break;
    case EBADF:
	reason = "invalid file descriptor; program error";
	break;
    case EBADMSG:
	/* Should this be fatal? */
	reason = "the message on the stream wasn't data";
	break;
    case EDEADLK:
	reason = "a deadlock would have happened [EDEADLK]";
	break;
    case EFAULT:
	if (buffer == (char *) 0) {
	    reason = "can't read into a null buffer; program error";
	    ewhat = lrqE_EPROG;
	} else {
	    reason = "buffer points into outer space; program error";
	    ewhat = lrqE_EPROG;
	}
	break;
    case EINTR:
	/* Programs dealing with signals should deal with this. */
	if (ewhat == lrqE_ENORMAL) {
	    return -1;
	}
	reason = "interrupted by a signal";
	break;
    case EIO:
	reason = "there may have been a hardware problem [EIO]";
	break;
    case ENXIO:
	reason = "attempt to read past the end of the device";
	break;
    case EINVAL:
	reason = "can't read from a stream linked to a multiplexor";
	break;
    case ENOLCK:
	reason = "the system record lock table was full [ENOLCK]";
	break;
    case ENOLINK:
	reason = "the remote machine went down [ENOLINK]";
	break;
    case EMULTIHOP: /* supposedly detected on open()... */
	reason = "An impossible RFS bug has happened [EMULTIHOP]";
	break;
    default:
	if (errno < sys_nerr) {
	    reason = sys_errlist[errno];
	} else {
	    static char buf[100];

	    sprintf(buf, "unknown error (errno = %d), see intro(2)", errno);
	    reason = buf;
	}
	ewhat = lrqE_EINTERNAL;
    }

    /* Probably you don't want this to be fatal... */
    lrqE_error(ewhat, "failed to read %u bytes from fd(%d) --\n\t%s\n",
		    nbytes, descriptor, reason);
}

#ifdef lrqEwrite
# undef lrqEwrite
#endif
#undef write

int
lrqEwrite(descriptor, buffer, nbytes)
    int descriptor;
    char *buffer;
    unsigned int nbytes;
{
    int retval;
    int ewhat;
    int savsyserrno;
    char *reason;
    int flags;

    errno = 0;

    if ((retval = write(descriptor, buffer, nbytes)) >= 0) {
	return retval;
    }
    /* we have an error */

    /* Save the original error message */
    savsyserrno = errno;

    ewhat = lrqE_severity;

    switch (errno) {
    case EAGAIN: /* not a fatal error at all! */
	if (fcntl(descriptor, F_GETFL, 0) & O_NDELAY) {
	    return -1;
	    /* If this happens, it is because you are using file locks
	     * (see chmod(2)) with NDELAY set, and you ought to be expecting it!
	     */
	}
	reason = "not enough internal system memory [EAGAIN]";
	break;
    case EBADF:
	reason = "fd is not open (or not for writing); program error";
	break;
    case EDEADLK:
	reason = "a deadlock would have happened [EDEADLK]";
	break;
    case EFAULT:
	if (buffer == (char *) 0) {
	    reason = "can't write from a null buffer; program error";
	} else {
	    reason = "buffer points into outer space; program error";
	}
	ewhat = lrqE_EPROG;
	break;
    case EFBIG:
	reason = "the resulting file would be too big [EFBIG]";
	break;
    case EINTR:
	/* I am not sure if I should do this... but it's probably OK */
	if (!isatty(descriptor)) {
	    if ((retval = write(descriptor, buffer, nbytes)) >= 0) {
		return retval;
	    }
	}
	errno = savsyserrno;
	reason = "the write was interrupted by a signal";
	break;
    case EINVAL:
	reason = "you can't write to a stream linked to a multiplexer";
	ewhat = lrqE_EPROG;
	break;
    case ENOLCK:
	reason = "the system record lock table was full [ENOLCK]";
	break;
    case EIO: /* only with O_SYNC normally, I guess! */
	reason = "there may have been a hardware problem [EIO]";
	break;
    case ENXIO:
	reason = "there was a hangup on the stream";
	/* Can someone translate that into English please? -- Lee */
	break;
    case ENOLINK:
	reason = "the remote machine went down [ENOLINK]";
	break;
    case EMULTIHOP: /* supposedly detected on open()... */
	reason = "An impossible RFS bug has happened [EMULTIHOP]";
	ewhat = lrqE_EPROG;
	break;
    case ENOSPC:
	reason = "the file system is full";
	break;
    case EPIPE:
	reason = "writing to a pipe with no-one listening [EPIPE]";
	break;
    case ERANGE:
	reason = "can't write that amount to this stream";
	break;
    default:
	if (errno < sys_nerr) {
	    reason = sys_errlist[errno];
	} else {
	    static char buf[100];

	    sprintf(buf, "unknown error (errno = %d), see intro(2)", errno);
	    reason = buf;
	}
	ewhat = lrqE_EINTERNAL;
    }

    /* Possibly you don't want this to be fatal... */
    lrqE_error(ewhat, "failed to write %u bytes to fd(%d) --\n\t%s\n",
		    nbytes, descriptor, reason);
}
