#include "lrqElib.h"
#include "lrqEdefs.h"
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

extern char lrqEwhat[], lrqEwhy[];

#ifdef lrqEread
# undef lrqEread
#endif
#undef read

/* If I was really cunning, I could save a list of paths
 * to correspond to opened files.  This would work in most cases,
 * since most read errors seem to happen on files that were
 * explicitly opened, but it would make errors on inherited or dup()'d
 * descriptors somewhat misleading, especially in the face of
 * link, unlink, and fcntl, which can also change the name of an open file!
 */

int
lrqEread(descriptor, buffer, nbytes)
    int descriptor;
    char *buffer;
    unsigned int nbytes;
{
    int retval;
    int savsyserrno;
    char *why;
    int ewhat;
    int flags;

    errno = 0;

    if ((retval = read(descriptor, buffer, nbytes)) >= 0) {
	return retval;
    }
    /* we have an error */

    /* by default, don't exit from read() */
    ewhat = lrqE_severity & (~lrqE_EFATAL);

    /* Save the original error message */
    savsyserrno = errno;

    switch (errno) {
    case EAGAIN: /* not a fatal error at all! */
	if (fcntl(descriptor, F_GETFL, 0) & O_NDELAY) {
	    return -1;
	    /* If this happens, it is because you are using file locks
	     * (see chmod(2)) with NDELAY set, and you ought to be expecting it!
	     */
	}
	why = "not enough internal system memory [EAGAIN]";
	break;
    case EBADF:
	why = "invalid file descriptor; program error";
	break;
    case EBADMSG:
	/* Should this be fatal? */
	why = "the message on the stream wasn't data";
	break;
    case EDEADLK:
	why = "a deadlock would have happened [EDEADLK]";
	break;
    case EFAULT:
	if (buffer == (char *) 0) {
	    why = "can't read into a null buffer; program error";
	    ewhat = lrqE_EPROG;
	} else {
	    why = "buffer points into outer space; program error";
	    ewhat = lrqE_EPROG;
	}
	break;
    case EINTR:
	/* Programs dealing with signals should deal with this. */
	if (ewhat == lrqE_ENORMAL) {
	    return -1;
	}
	why = "interrupted by a signal";
	break;
    case EIO:
	why = "there may have been a hardware problem [EIO]";
	break;
    case ENXIO:
	why = "attempt to read past the end of the device";
	break;
    case EINVAL:
	why = "can't read from a stream linked to a multiplexor";
	break;
    case ENOLCK:
	why = "the system record lock table was full [ENOLCK]";
	break;
    case ENOLINK:
	why = "the remote machine went down [ENOLINK]";
	break;
    case EMULTIHOP: /* supposedly detected on open()... */
	why = "An impossible RFS bug has happened [EMULTIHOP]";
	break;
    default:
	if (errno < sys_nerr) {
	    why = sys_errlist[errno];
	} else {
	    why = lrqEwhy;
	    sprintf(why, "unknown error (errno = %d), see intro(2)", errno);
	}
	ewhat = lrqE_EINTERNAL;
    }

    /* Probably you don't want this to be fatal... */
    lrqE__setwhat("failed to read %u bytes from file descriptor %d",
							nbytes, descriptor);
    if (why != lrqEwhy) lrqE__setwhy(why);

    lrqError(ewhat, "read", "read", lrqEwhat, lrqEwhy);

    return -1;
}

#ifdef lrqEwrite
# undef lrqEwrite
#endif
#undef write

int
lrqEwrite(descriptor, buffer, nbytes)
    int descriptor;
    char *buffer;
    unsigned int nbytes;
{
    int retval;
    int ewhat;
    int savsyserrno;
    char *why;
    int flags;

    errno = 0;

    if ((retval = write(descriptor, buffer, nbytes)) >= 0) {
	return retval;
    }
    /* we have an error */

    /* Save the original error message */
    savsyserrno = errno;

    ewhat = lrqE_severity;

    switch (errno) {
    case EAGAIN: /* not a fatal error at all! */
	if (fcntl(descriptor, F_GETFL, 0) & O_NDELAY) {
	    return -1;
	    /* If this happens, it is because you are using file locks
	     * (see chmod(2)) with NDELAY set, and you ought to be expecting it!
	     */
	}
	why = "not enough internal system memory [EAGAIN]";
	break;
    case EBADF:
	why = "fd is not open (or not for writing); program error";
	break;
    case EDEADLK:
	why = "a deadlock would have happened [EDEADLK]";
	break;
    case EFAULT:
	if (buffer == (char *) 0) {
	    why = "can't write from a null buffer; program error";
	} else {
	    why = "buffer points into outer space; program error";
	}
	ewhat = lrqE_EPROG;
	break;
    case EFBIG:
	why = "the resulting file would be too big [EFBIG]";
	break;
    case EINTR:
	/* I am not sure if I should do this... but it's probably OK */
	if (!isatty(descriptor)) {
	    if ((retval = write(descriptor, buffer, nbytes)) >= 0) {
		return retval;
	    }
	}
	errno = savsyserrno;
	why = "the write was interrupted by a signal";
	break;
    case EINVAL:
	why = "you can't write to a stream linked to a multiplexer";
	ewhat = lrqE_EPROG;
	break;
    case ENOLCK:
	why = "the system record lock table was full [ENOLCK]";
	break;
    case EIO: /* only with O_SYNC normally, I guess! */
	why = "there may have been a hardware problem [EIO]";
	break;
    case ENXIO:
	why = "there was a hangup on the stream";
	/* Can someone translate that into English please? -- Lee */
	break;
    case ENOLINK:
	why = "the remote machine went down [ENOLINK]";
	break;
    case EMULTIHOP: /* supposedly detected on open()... */
	why = "An impossible RFS bug has happened [EMULTIHOP]";
	ewhat = lrqE_EPROG;
	break;
    case ENOSPC:
	why = "the file system is full";
	break;
    case EPIPE:
	why = "writing to a pipe with no-one listening [EPIPE]";
	break;
    case ERANGE:
	why = "can't write that amount to this stream";
	break;
    default:
	if (errno < sys_nerr) {
	    why = sys_errlist[errno];
	} else {
	    sprintf(lrqEwhy, "unknown error (errno = %d), see intro(2)", errno);
	    why = lrqEwhy;
	}
	ewhat = lrqE_EINTERNAL;
    }

    lrqE__setwhat("failed to write %u bytes to file descriptor %d",
							nbytes, descriptor);
    if (why != lrqEwhy) lrqE__setwhy(why);
    lrqE_error(ewhat, "open", "open", lrqEwhat, lrqEwhy);

    return -1;
}
