/* Part of Liam Quin's Error Checking Libary.
 * Use this at your own peril.
 */

/* Let's get the real versions of everything */
#undef lrqE
#include "lrqElib.h"
#include "lrqEdefs.h"

#include <sys/types.h> /* needed by stat.h */
#include <sys/stat.h> /* for isdir() */

/* let's get the real versions */
#include <malloc.h>

/* Echeckpath(path, errno, mode) -- check for a problem in the given path.
 * We check for --
 * @  a component not being a directory (ENOTDIR)
 * @  a component denies search (execute) permission
 * @  you don't have write access to the parent directory for an
 *    operation that erquires it (e.g. rm /bin/sh when not root or bin)
 *
 * path -- a filename or path ("martin" or "/bin/sh" or "../jfklsf" ...)
 *
 * errno -- the value of errno after the system call returned an error
 *
 * mode -- a number, containing one or more of the following:
 *
 * O_RDWR, O_RDONLY, O_CREAT -- what we wanted to do to the file
 *
 * In addition, you can use O_UNLINK, O_EXEC and O_STAT, even if they
 * are not defined in your <file.h> or <fcntl.h>.
 *
 * TODO:
 * Maybe we could check for typos or spelling mistakes here, but see
 * ../doc/philosophy for reasons why *not* to do this!  In short, though,
 * it would be very irritating to say ``cd /bun'' and be told that you
 * meant /bin; people would say ``well if it know what I meant, why doesn't
 * it go there?''...
 * 
 */

char *
lrqEcheckpath(path, error, mode)
    char *path; /* the file we couldn't access */
    int error; /* what the system told us */
    int mode;  /* what we tried to do to it */
{
    char *p;
    int n_tried = 0;
    struct stat parent;
    char *pp = 0; /* parent pointer */

    /* First see if all of the elements in the path are directories.
     */
    for (p = path; *p; p++) {
	/* NOTE: don't check the file itself! */
	if (*p == '/' && p > path) {
	    char *boy;
	    char *lrqE_perms();
	    int id;
	    n_tried++;
	    *p = '\0';
	    pp = p; /* in case this was the parent directory */

again:
	    if ((id = lrqE_isdir(path)) < 0) {
		/* Well, we can't stat() this directory...
		 * so maybe it doesn't exist, or maybe we don't have
		 * permission
		 */
		switch (errno) {
		case ENOENT:
		    sprintf(lrqE_whybuf, "\"%s\" does not exist", path);
		    *p = '/';
		    break;
		case EACCESS:
		    sprintf(lrqE_whybuf, "can't search \"%s\"", path);
		    *p = '/';
		    break;
		case EINTR:
		    goto again;
		case EMULTIHOP:
		case ENOLINK:
		    sprintf(lrqE_whybuf, "RFS error on \"%s\"", path);
		    *p = '/';
		/* NOTE: we don't look for EFAULT, etc., because that
		 * would represent a bug in the error routines, which
		 * we couldn't report without blushing... and the of
		 * whose existence the user is presumably unaware!
		 */
		default: /* who knows? */
		    sprintf(lrqE_whybuf,
		     "problem %d reading \"%s\" [%s]", path,
		     (errno < sys_nerr) ? sys_errlist[errno] : "unknown");
		    *p = '/';
		    break;
		}
		return lrqE_whybuf;
	    } else if (id == 0) {
		sprintf(lrqE_whybuf, "\"%s\" isn't a directory.", path);
		*p = '/';	/* restore the damage... */
		return lrqE_whybuf;
	    } else if ((boy = lrqE_perms(path, mode)) != (char *) 0) {
		/* Check permissions */
		if (boy != lrqE_whybuf) {
		    (void) sprintf(lrqE_whybuf, boy, path);
		}
		*p = '/';
		return lrqE_whybuf;
	    }

	    /* restore the damage... */
	    *p = '/';
	} /* end if */
    } /* end for */

    /* Hmph.  Now check the parent directory */
    if (!n_tried) {
	if (stat("..", &parent) < 0) {
	    return "there isn't a parent directory (..)";
	} 
    } else {
	*pp = '\0';
	if (stat(path, &parent) < 0) { /* maybe someone did an rm -r */
	    sprintf(lrqE_whybuf, "\"%s\" went away", path);
	    *p = '/';
	    return lrqE_whybuf;
	}
    }
    /* Well, we didn't find the problem */
    switch (error) {
    case EACCES:
    return strcpy(lrqE_whybuf, "permission (or directory search) denied");
    case ENOENT:
    return strcpy(lrqE_whybuf, "a component of the path isn't there");
    case ENOTDIR:
    return strcpy(lrqE_whybuf, "a component of the path isn't a directory");;
    default:
    (void) strcpy(lrqE_whybuf, "unknown error");
    return (char *) 0;
    }
    /*NOTREACHED*/
}

int
lrqE_isdir(path)
    char *path;
{
    struct stat s;

    if (stat(path, &s) < 0) {
	return -1;
    }
    return (s.st_mode & S_IFDIR) ? 1 : 0;
}

char *
lrqE_perms(path, mode)
    char *path;
    char *mode;
{
    int m;

    if (path == (char *) 0) {
	return "[corrupt pathname]";
    }
    if (access(path, 01) < 0) {
	return "no permission to search directory %s";
    }
    if (access(path, 04) < 0) {
	return "can't read directory %s";
    }
    return (char *) 0;
}
